<?php

/**
  * SquirrelMail Login Authentication Plugin
  *
  * Copyright (c) 2004-2012 Paul Lesniewski <paul@squirrelmail.org>,
  * Copyright (c) 2001 Tyler Akins
  *
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage login_auth
  *
  */



/**
  * Initialize this plugin (load config values)
  *
  * @param boolean $debug When TRUE, do not suppress errors when including
  *                       configuration files (OPTIONAL; default FALSE)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function login_auth_init($debug=FALSE)
{

   if ($debug)
   {
      if (!include_once(SM_PATH . 'config/config_login_auth.php'))
         if (!include_once(SM_PATH . 'plugins/login_auth/config.php'))
            if (!include_once(SM_PATH . 'plugins/login_auth/config_default.php'))
               return FALSE;
   }
   else
   {
      if (!@include_once(SM_PATH . 'config/config_login_auth.php'))
         if (!@include_once(SM_PATH . 'plugins/login_auth/config.php'))
            if (!@include_once(SM_PATH . 'plugins/login_auth/config_default.php'))
               return FALSE;
   }


   return TRUE;

}



/**
  * Change the signout page
  *
  */
function login_auth_logout()
{

   global $use_alternate_signout_page, $signout_page;
   login_auth_init();
   $used_external_auth = FALSE;
   sqgetGlobalVar('used_external_auth_creds_for_login', $used_external_auth, SQ_SESSION);


   // note we won't do anything if the user didn't use external
   // authentication to log in or this feature is turned off
   //
   if (!$used_external_auth || !$use_alternate_signout_page) return;


   else if ($use_alternate_signout_page !== 1)
   {
      if ($signout_page == '')
         $signout_page = get_location() . '/signout.php';
      header('Location: ' . str_replace(array('%s', '%e'),
                                        array($signout_page, urlencode($signout_page)),
                                        $use_alternate_signout_page));
      exit();
   }


   // otherwise, we'll use our special logout page
   //
   $signout_page = SM_PATH . 'plugins/login_auth/logout.php';

}



/**
  * Change the login link on error pages
  *
  */
function login_auth_error_login_link(&$args)
{
   global $external_session_expiration_relogin_link;
   login_auth_init();

   if ($external_session_expiration_relogin_link)
   {
      if (check_sm_version(1, 5, 2))
         $args[2]['URI'] = $external_session_expiration_relogin_link;
      else
         $args[3] = $external_session_expiration_relogin_link;
   }
}



/**
  * Verifies that an externally logged in user is still logged in
  * based on the presence of a server environment variable.  If
  * said variable is not found or is empty, the user will be logged
  * out.
  *
  */
function login_auth_verify_login()
{
//TODO: We could check if this is the login page and just bail, but the session *should* have been destroyed in that case, so doing so should only fix problems that might need to be addressed elsewhere

   global $required_environment_variable,
          $required_environment_variable_value_type,
          $required_environment_variable_value;
   login_auth_init();


   // just bail if feature isn't being used
   //
   if (empty($required_environment_variable)) return;


   $used_external_auth = FALSE;
   sqgetGlobalVar('used_external_auth_creds_for_login', $used_external_auth, SQ_SESSION);


   if ($required_environment_variable_value_type == 2)
   {
      $required_environment_variable_value = NULL;
      sqgetGlobalVar('external_auth_required_environment_variable_value', $required_environment_variable_value, SQ_SESSION);
   }


   // make sure our needed environment variable is there (but only
   // if user was externally authenticated in the first place)
   //
   if ($used_external_auth)
   {

      list($environment_variable, $ignore) = get_credentials_from_environment($required_environment_variable, 'ignore');


      // did the needed environment variable go away or change?
      //
      if (($required_environment_variable_value_type == 1 && empty($environment_variable))
        || ($required_environment_variable_value_type != 1 && $environment_variable != $required_environment_variable_value))
      {
         header('Location: ' . sqm_baseuri() . 'plugins/login_auth/expired.php');
         exit;
      }

   }

}



/** 
  * Make sure to use alternative auth credentials when
  * logging in if needed
  *
  */
function login_auth_login_before()
{

   global $plugins, $login_auth, $login_auth_user, $login_auth_pass,
          $just_logged_in, $login_username, $secretkey;

   sqgetGlobalVar('login_auth', $login_auth, SQ_SESSION);
   sqgetGlobalVar('login_auth_user', $login_auth_user, SQ_SESSION);
   sqgetGlobalVar('login_auth_pass', $login_auth_pass, SQ_SESSION);


   // user is logging in via normal means
   //
   if (empty($login_auth) || $login_auth !== 'YES')
   {
      global $normal_login_behavior;
      login_auth_init();

      // we really shouldn't get here if $normal_login_behavior
      // is such that we have to send the user away, but let's
      // cover our bases
      //
      if (!$normal_login_behavior)
      {
         sq_change_text_domain('login_auth');
         logout_error(_("Please log in first"));
         exit();
      }
      else if ($normal_login_behavior !== 1)
      {
         $location = get_location();
         $location = substr($location, 0, strrpos($location, '/')) . '/src/login.php';
         header('Location: ' . str_replace(array('%s', '%e'),
                                           array($location, urlencode($location)),
                                           $normal_login_behavior));
         exit();
      }

      // allow normal login
      //
      return;
   }


   // lets this plugin detect at any time later if external
   // auth credentials were used to log in
   //
   sqsession_register('yes', 'used_external_auth_creds_for_login');


   $login_username = $login_auth_user;
   $secretkey = $login_auth_pass;
   $just_logged_in = 1;


   // adjust for password forget plugin
   //
   if (in_array('password_forget', $plugins)) 
   {
      $login_username = 'login_auth_user';
      $secretkey = 'login_auth_pass';

      // in case password forget has already been called...
      //
      if (function_exists('password_forget_post_do')) password_forget_post_do();

   }

}



/**
  * Check if user has already authenticated themselves; 
  * can we skip login page?
  *
  */
function login_auth_skip_login()
{

   $credentials = login_auth_validate_credentials();
   if ($credentials !== FALSE)
   {
      global $login_auth_user, $login_auth_pass, $login_auth,
             $required_environment_variable, $required_environment_variable_value_type;

      // stash away external login session validation variable value for later
      //
      if (!empty($required_environment_variable)
       && $required_environment_variable_value_type == 2)
      {
         list($environment_variable, $ignore) = get_credentials_from_environment($required_environment_variable, 'ignore');
         sqsession_register($environment_variable, 'external_auth_required_environment_variable_value');
      }

      list($login_auth_user, $login_auth_pass) = $credentials;
      $login_auth = 'YES';
      sqsession_register($login_auth, 'login_auth');
      sqsession_register($login_auth_user, 'login_auth_user');
      sqsession_register($login_auth_pass, 'login_auth_pass');
      header('Location: ' . SM_PATH . 'src/redirect.php?login_username=login_auth');
      exit();
   }


   // the user isn't externally authenticated - so what do we do?
   //
   global $normal_login_behavior;
   login_auth_init();
   if (!$normal_login_behavior)
   {
      sq_change_text_domain('login_auth');
      logout_error(_("Please log in first"));
      exit();
   }
   else if ($normal_login_behavior !== 1)
   {
      $location = get_location();
      $location = substr($location, 0, strrpos($location, '/')) . '/src/login.php';
      header('Location: ' . str_replace(array('%s', '%e'),
                                        array($location, urlencode($location)),
                                        $normal_login_behavior));
      exit();
   }

   // otherwise, continue to normal login page...

}



/**
  * Detect if external user authentication information is
  * available and if it is, optionally validate the user
  * against the IMAP server.
  *
  * @param boolean $no_login_attempt When TRUE, do not try to
  *                                  validate the username and
  *                                  password against the IMAP
  *                                  server; only check if they
  *                                  are available (OPTIONAL;
  *                                  default FALSE)
  *
  * @return mixed A two element array (username and password (in that
  *               order) to use from now on to authenticate against
  *               the IMAP server) if a user is found to have been
  *               pre-authenticated externally, or FALSE otherwise.
  *
  */
function login_auth_validate_credentials($no_login_attempt=FALSE)
{

   global $authenticated_username_location, $authenticated_password_location,
          $external_auth_validation_type, $authenticated_saml_compress_assertion;
   login_auth_init();


   list($AUTH_USER, $AUTH_PW) = get_credentials_from_environment($authenticated_username_location, $authenticated_password_location);


   if ($external_auth_validation_type == 'http_auth'
    || $external_auth_validation_type == 'authenticated_saml'
    || $external_auth_validation_type == 'trusted_saml')
   {

      // no user or pwd (or token)?  return FALSE
      // (trusted saml module can have an empty password)
      //
      if (empty($AUTH_USER)
       || (empty($AUTH_PW) && $external_auth_validation_type !== 'trusted_saml'))
         return FALSE;

      // set up a trusted authentication to log the user in
      //
      if ($external_auth_validation_type == 'trusted_saml')
      {
         global $trusted_saml_username, $trusted_saml_password;
         sqsession_register($trusted_saml_username, 'authz');
         $AUTH_PW = $trusted_saml_password;
//NOTE: can hack $AUTH_USER here to fake out/test "trusted_saml"
      }

      if ($external_auth_validation_type == 'authenticated_saml'
       && $authenticated_saml_compress_assertion)
      {
         $AUTH_PW = base64_encode(gzcompress(base64_decode($AUTH_PW)));
      }

      // (HTTP auth needs no futher actions)

   }

   // call external validation routine
   //
   else
   {
      $credentials = $external_auth_validation_type($AUTH_USER, $AUTH_PW);
      if (!$credentials) return FALSE;
      list($AUTH_USER, $AUTH_PW) = $credentials;
   }


   // if the Login Manager plugin is in use, let it massage the username
   //
   global $plugins;
   if (in_array('vlogin', $plugins))
   {
      include_once(SM_PATH . 'plugins/vlogin/functions.php');
      global $login_username;
      $login_username = $AUTH_USER;
      vlogin_domain_do(NULL);
      $AUTH_USER = $login_username;
      $login_username = NULL;
   }


   // if don't need to check with IMAP, return now
   //
   if ($no_login_attempt) return array($AUTH_USER, $AUTH_PW);


   // try to log in with http-authenticated user info
   //
   include_once(SM_PATH . 'functions/imap.php');
   global $imapServerAddress, $imapPort, $onetimepad;


   $onetimepad = OneTimePadCreate(strlen($AUTH_PW));
   $key = OneTimePadEncrypt($AUTH_PW, $onetimepad);
   $imapConn = login_auth_sqimap_login($AUTH_USER, $key, $imapServerAddress, $imapPort, 1);


   // if we have a valid IMAP login, the credentials are good
   //
   if ($imapConn)
   {
      sqimap_logout($imapConn);
      return array($AUTH_USER, $AUTH_PW);
   }


   // or if not...
   //
   return FALSE;

}



/**
  * Extracts user credentials from the server environment
  *
  * NOTE that this function does not guarantee that either
  *      the username or password won't be empty - the caller
  *      is responsible for validating needed data
  *
  * @param string $username_location The location in the web server
  *                                  environment to search for the
  *                                  username variable
  * @param string $password_location The location in the web server
  *                                  environment to search for the
  *                                  password variable
  *
  * @return array A two-element array, the first value being
  *               the username and the second element being
  *               the password
  *
  */
function get_credentials_from_environment($username_location, $password_location)
{

   if (is_array($username_location))
   {
      foreach ($username_location as $var_name)
      {
         $AUTH_USER = '';
         sqgetGlobalVar($var_name, $AUTH_USER, SQ_SERVER);
         if (!empty($AUTH_USER))
            break;
      }
   }
   else
   {
      $AUTH_USER = '';
      sqgetGlobalVar($username_location, $AUTH_USER, SQ_SERVER);
   }


   if (is_array($password_location))
   {
      foreach ($password_location as $var_name)
      {
         $AUTH_PW = '';
         sqgetGlobalVar($var_name, $AUTH_PW, SQ_SERVER);
         if (!empty($AUTH_PW))
            break;
      }
   }
   else
   {
      $AUTH_PW = '';
      sqgetGlobalVar($password_location, $AUTH_PW, SQ_SERVER);
   }

   return array($AUTH_USER, $AUTH_PW);

}



/**
  * Logs the user into the imap server.  If $hide is set, no error messages
  * will be displayed.  This function returns the imap connection handle.
  *
  * Stolen from functions/imap_general.php; we need our own version because
  * the one we took it from will only exit or show a login error page when
  * we have a bad login.  Instead, we only want to return FALSE in such a
  * case.
  *
  * At some point in the evolution of SquirrelMail 1.5.x, this should no
  * longer become necessary.
  *
  */
function login_auth_sqimap_login($username, $password, $imap_server_address, $imap_port, $hide)
{
    global $color, $squirrelmail_language, $onetimepad, $use_imap_tls, $imap_auth_mech,
           $sqimap_capabilities;

    // Note/TODO: This hack grabs the $authz argument from the session
    $authz = '';
    global $authz;
    sqgetglobalvar('authz' , $authz , SQ_SESSION);

    if(!empty($authz)) {
        /* authz plugin - specific:
         * Get proxy login parameters from authz plugin configuration. If they
         * exist, they will override the current ones.
         * This is useful if we want to use different SASL authentication mechanism
         * and/or different TLS settings for proxy logins. */
        global $authz_imap_auth_mech, $authz_use_imap_tls, $authz_imapPort_tls;
        $imap_auth_mech = !empty($authz_imap_auth_mech) ? strtolower($authz_imap_auth_mech) : $imap_auth_mech;
        $use_imap_tls = !empty($authz_use_imap_tls)? $authz_use_imap_tls : $use_imap_tls;
        $imap_port = !empty($authz_use_imap_tls)? $authz_imapPort_tls : $imap_port;

        if($imap_auth_mech == 'login' || $imap_auth_mech == 'cram-md5') {
            logout_error("Misconfigured Plugin (authz or equivalent):<br/>".
            "The LOGIN and CRAM-MD5 authentication mechanisms cannot be used when attempting proxy login.");
            exit;
        }
    }

    if (!isset($onetimepad) || empty($onetimepad)) {
        sqgetglobalvar('onetimepad' , $onetimepad , SQ_SESSION );
    }
    $imap_server_address = sqimap_get_user_server($imap_server_address, $username);
        $host=$imap_server_address;

        if (($use_imap_tls == true) and (check_php_version(4,3)) and (extension_loaded('openssl'))) {
          /* Use TLS by prefixing "tls://" to the hostname */
          $imap_server_address = 'tls://' . $imap_server_address;
        }

    $imap_stream = @fsockopen($imap_server_address, $imap_port, $error_number, $error_string, 15);

    /* Do some error correction */
    if (!$imap_stream) {
        if (!$hide) {
            set_up_language($squirrelmail_language, true);
            require_once(SM_PATH . 'functions/display_messages.php');
            logout_error( sprintf(_("Error connecting to IMAP server: %s."), $imap_server_address).
                "<br />\r\n$error_number : $error_string<br />\r\n",
                sprintf(_("Error connecting to IMAP server: %s."), $imap_server_address) );
        }
        // CHANGE FOR LOGIN_AUTH PLUGIN:
        //exit;
        return FALSE;
    }

    $server_info = fgets ($imap_stream, 1024);

    /* Decrypt the password */
    $password = OneTimePadDecrypt($password, $onetimepad);

    if (!isset($sqimap_capabilities)) {
        sqgetglobalvar('sqimap_capabilities' , $sqimap_capabilities , SQ_SESSION );
    }

    if (($imap_auth_mech == 'cram-md5') OR ($imap_auth_mech == 'digest-md5')) {
        // We're using some sort of authentication OTHER than plain or login
        $tag=sqimap_session_id(false);
        if ($imap_auth_mech == 'digest-md5') {
            $query = $tag . " AUTHENTICATE DIGEST-MD5\r\n";
        } elseif ($imap_auth_mech == 'cram-md5') {
            $query = $tag . " AUTHENTICATE CRAM-MD5\r\n";
        }
        fputs($imap_stream,$query);
        $answer=sqimap_fgets($imap_stream);
        // Trim the "+ " off the front
        $response=explode(" ",$answer,3);
        if ($response[0] == '+') {
            // Got a challenge back
            $challenge=$response[1];
            if ($imap_auth_mech == 'digest-md5') {
                $reply = digest_md5_response($username,$password,$challenge,'imap',$host,$authz);
            } elseif ($imap_auth_mech == 'cram-md5') {
                $reply = cram_md5_response($username,$password,$challenge);
            }
            fputs($imap_stream,$reply);
            $read=sqimap_fgets($imap_stream);
            if ($imap_auth_mech == 'digest-md5') {
                // DIGEST-MD5 has an extra step..
                if (substr($read,0,1) == '+') { // OK so far..
                    fputs($imap_stream,"\r\n");
                    $read=sqimap_fgets($imap_stream);
                }
            }
            $results=explode(" ",$read,3);
            $response=$results[1];
            $message=$results[2];
        } else {
            // Fake the response, so the error trap at the bottom will work
            $response="BAD";
            $message='IMAP server does not appear to support the authentication method selected.';
            $message .= '  Please contact your system administrator.';
        }
    } elseif ($imap_auth_mech == 'login') {
        // this is a workaround to alert users of LOGINDISABLED, which is done "right" in
        // devel but requires functions not available in stable. RFC requires us to
        // not send LOGIN when LOGINDISABLED is advertised.
        if(stristr($server_info, 'LOGINDISABLED')) {
            $response = 'BAD';
            $message = _("The IMAP server is reporting that plain text logins are disabled.").' '.
                _("Using CRAM-MD5 or DIGEST-MD5 authentication instead may work.").' ';
            if (!$use_imap_tls) {
                $message .= _("Also, the use of TLS may allow SquirrelMail to login.").' ';
            }
            $message .= _("Please contact your system administrator and report this error.");
        } else {
            // Original IMAP login code
            if(sq_is8bit($username) || sq_is8bit($password)) {
                $query['commands'][0] = 'LOGIN';
                $query['literal_args'][0] = $username;
                $query['commands'][1] = '';
                $query['literal_args'][1] = $password;
                $read = sqimap_run_literal_command($imap_stream, $query, false, $response, $message);
            } else {
                $query = 'LOGIN "' . quoteimap($username) . '"'
                       . ' "' . quoteimap($password) . '"';
                $read = sqimap_run_command ($imap_stream, $query, false, $response, $message);
            }
        }
    } elseif ($imap_auth_mech == 'plain') {
        /***
         * SASL PLAIN, RFC 4616 (updates 2595)
         *
         * The mechanism consists of a single message, a string of [UTF-8]
         * encoded [Unicode] characters, from the client to the server.  The
         * client presents the authorization identity (identity to act as),
         * followed by a NUL (U+0000) character, followed by the authentication
         * identity (identity whose password will be used), followed by a NUL
         * (U+0000) character, followed by the clear-text password.  As with
         * other SASL mechanisms, the client does not provide an authorization
         * identity when it wishes the server to derive an identity from the
         * credentials and use that as the authorization identity.
         */
        $tag=sqimap_session_id(false);
        $sasl = (isset($sqimap_capabilities['SASL-IR']) && $sqimap_capabilities['SASL-IR']) ? true : false;
        if(!empty($authz)) {
            $auth = base64_encode("$username\0$authz\0$password");
        } else {
            $auth = base64_encode("$username\0$username\0$password");
        }
        if ($sasl) {
            // IMAP Extension for SASL Initial Client Response
            // <draft-siemborski-imap-sasl-initial-response-01b.txt>
            $query = $tag . " AUTHENTICATE PLAIN $auth\r\n";
            fputs($imap_stream, $query);
            $read = sqimap_fgets($imap_stream);
        } else {
            $query = $tag . " AUTHENTICATE PLAIN\r\n";
            fputs($imap_stream, $query);
            $read=sqimap_fgets($imap_stream);
            if (substr($read,0,1) == '+') { // OK so far..
                fputs($imap_stream, "$auth\r\n");
                $read = sqimap_fgets($imap_stream);
            }
        }
        $results=explode(" ",$read,3);
        $response=$results[1];
        $message=$results[2];

    } else {
        $response="BAD";
        $message="Internal SquirrelMail error - unknown IMAP authentication method chosen.  Please contact the developers.";
    }

    /* If the connection was not successful, lets see why */
    if ($response != 'OK') {
        if (!$hide) {
            if ($response != 'NO') {
                /* "BAD" and anything else gets reported here. */
                $message = htmlspecialchars($message);
                set_up_language($squirrelmail_language, true);
                require_once(SM_PATH . 'functions/display_messages.php');
                if ($response == 'BAD') {
                    $string = sprintf (_("Bad request: %s")."<br />\r\n", $message);
                } else {
                    $string = sprintf (_("Unknown error: %s") . "<br />\n", $message);
                }
                if (isset($read) && is_array($read)) {
                    $string .= '<br />' . _("Read data:") . "<br />\n";
                    foreach ($read as $line) {
                        $string .= htmlspecialchars($line) . "<br />\n";
                    }
                }
                error_box($string,$color);
                exit;
            } else {
                /*
                 * If the user does not log in with the correct
                 * username and password it is not possible to get the
                 * correct locale from the user's preferences.
                 * Therefore, apply the same hack as on the login
                 * screen.
                 *
                 * $squirrelmail_language is set by a cookie when
                 * the user selects language and logs out
                 */

                set_up_language($squirrelmail_language, true);
                include_once(SM_PATH . 'functions/display_messages.php' );
                sqsession_destroy();
                /* terminate the session nicely */
                sqimap_logout($imap_stream);
                logout_error( _("Unknown user or password incorrect.") );
                exit;
            }
        } else {
            // CHANGE FOR LOGIN_AUTH PLUGIN:
            //exit;
            return FALSE;
        }
    }
    return $imap_stream;
}



